# Finance API - Backend Documentation

RESTful API untuk Finance App menggunakan PHP dan MySQL.

## Base URL

```
http://localhost/finance_api
```

## Endpoints

### Authentication

#### 1. Login
```
POST /auth/login.php

Request Body:
{
  "email": "user@example.com",
  "password": "password123"
}

Response:
{
  "success": true,
  "message": "Login berhasil",
  "data": {
    "id": 1,
    "name": "User Name",
    "email": "user@example.com",
    "token": "generated_token_here"
  }
}
```

#### 2. Register
```
POST /auth/register.php

Request Body:
{
  "name": "New User",
  "email": "newuser@example.com",
  "password": "password123"
}

Response:
{
  "success": true,
  "message": "Registrasi berhasil"
}
```

### Transactions

#### 1. Get Transactions
```
GET /transactions/get.php?user_id=1&start_date=2024-01-01&end_date=2024-12-31

Response:
{
  "success": true,
  "data": [
    {
      "id": 1,
      "user_id": 1,
      "type": "income",
      "amount": "5000000.00",
      "category": "Gaji",
      "description": "Gaji bulan Januari",
      "date": "2024-01-01",
      "created_at": "2024-01-01 10:00:00"
    }
  ]
}
```

#### 2. Add Transaction
```
POST /transactions/add.php

Request Body:
{
  "user_id": 1,
  "type": "expense",
  "amount": 150000,
  "category": "Makanan",
  "description": "Makan siang",
  "date": "2024-01-15"
}

Response:
{
  "success": true,
  "message": "Transaksi berhasil ditambahkan",
  "id": 123
}
```

#### 3. Update Transaction
```
PUT /transactions/update.php

Request Body:
{
  "id": 123,
  "user_id": 1,
  "type": "expense",
  "amount": 200000,
  "category": "Makanan",
  "description": "Makan siang + minum",
  "date": "2024-01-15"
}

Response:
{
  "success": true,
  "message": "Transaksi berhasil diupdate"
}
```

#### 4. Delete Transaction
```
DELETE /transactions/delete.php?id=123

Response:
{
  "success": true,
  "message": "Transaksi berhasil dihapus"
}
```

### Budgets

#### 1. Get Budgets
```
GET /budgets/get.php?user_id=1&month=2024-01

Response:
{
  "success": true,
  "data": [
    {
      "id": 1,
      "user_id": 1,
      "category": "Makanan",
      "planned_amount": "1000000.00",
      "spent_amount": "750000.00",
      "month": "2024-01",
      "created_at": "2024-01-01 10:00:00"
    }
  ]
}
```

#### 2. Add Budget
```
POST /budgets/add.php

Request Body:
{
  "user_id": 1,
  "category": "Transport",
  "planned_amount": 500000,
  "month": "2024-01"
}

Response:
{
  "success": true,
  "message": "Budget berhasil ditambahkan"
}
```

#### 3. Update Budget
```
PUT /budgets/update.php

Request Body:
{
  "id": 1,
  "user_id": 1,
  "category": "Transport",
  "planned_amount": 600000,
  "month": "2024-01"
}

Response:
{
  "success": true,
  "message": "Budget berhasil diupdate"
}
```

### Statistics

#### 1. Monthly Statistics
```
GET /statistics/monthly.php?user_id=1&start_month=2024-01&end_month=2024-06

Response:
{
  "success": true,
  "data": [
    {
      "month": "2024-01",
      "income": "5000000.00",
      "expense": "3500000.00"
    },
    {
      "month": "2024-02",
      "income": "5200000.00",
      "expense": "3800000.00"
    }
  ]
}
```

## Database Schema

### Users Table
```sql
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### Transactions Table
```sql
CREATE TABLE transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    type ENUM('income', 'expense') NOT NULL,
    amount DECIMAL(15,2) NOT NULL,
    category VARCHAR(50) NOT NULL,
    description TEXT,
    date DATE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

### Budgets Table
```sql
CREATE TABLE budgets (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    category VARCHAR(50) NOT NULL,
    planned_amount DECIMAL(15,2) NOT NULL,
    spent_amount DECIMAL(15,2) DEFAULT 0,
    month VARCHAR(7) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_category_month (user_id, category, month)
);
```

## Setup Instructions

1. Import database.sql ke MySQL
2. Konfigurasi database.php dengan kredensial MySQL Anda
3. Pastikan Apache dan MySQL running
4. Test endpoint dengan Postman atau browser

## Error Responses

```json
{
  "success": false,
  "message": "Error message here"
}
```

## Notes

- Semua endpoints mendukung CORS
- Password di-hash menggunakan bcrypt
- Spent amount di budget auto-update via database trigger
- Semua response dalam format JSON
